(*   
   Abstract domain based on the APRON library.

   Copyright (C) 2011 Antoine Miné
*)

open Datatypes
open Abstract_syntax
open Typed_syntax
open Semantics
open Domain
open Apron


(* GENERIC FUNCTOR *)
(* *************** *)

module type APRON_PARAM = sig 
  type lib
  val manager: lib Manager.t 
end


module Lin = Linearization.Make(Itv_rat)

module ApronDomain(Param: APRON_PARAM) = (struct

  let man = Param.manager

  type t = Param.lib Abstract1.t


  (**************************************************************************)
  (* OBSOLETE FUNCTIONS *)
  (* ****************** *)

  (* Texpr1 / Tcons1 functions *)
  (* we now use only Linexpr1 / Lincons1, and banal's internal linearization *)

(*
  (* TODO: bound-check ? *)
  let rec apron_of_expr ((e,t,x):expr typed) : Texpr1.expr =
    match e with
    | T_unary (op,e1) ->
        let e1 = apron_of_expr e1 in
        let r,d = apron_of_typ t in
        let e = match op with
        | A_UNARY_PLUS -> e1
        | A_UNARY_MINUS -> Texpr1.Unop (Texpr1.Neg, e1, r, d)
        | A_NOT -> invalid_arg "boolean node in Apron_domain.apron_of_expr"
        | A_cast _ -> Texpr1.Unop (Texpr1.Cast, e1, r, d)
        in
        e

    | T_binary (op,e1,e2) ->
        let e1, e2 = apron_of_expr e1, apron_of_expr e2 in
        let r,d = apron_of_typ t in
        let e = match op with
        | A_PLUS -> Texpr1.Binop (Texpr1.Add, e1, e2, r, d)
        | A_MINUS -> Texpr1.Binop (Texpr1.Sub, e1, e2, r, d)
        | A_MULTIPLY -> Texpr1.Binop (Texpr1.Mul, e1, e2, r, d)
        | A_DIVIDE -> Texpr1.Binop (Texpr1.Div, e1, e2, r, d)
        | A_MODULO -> Texpr1.Binop (Texpr1.Mod, e1, e2, r, d)
        | A_EQUAL | A_NOT_EQUAL | A_LESS | A_LESS_EQUAL | A_GREATER
        | A_GREATER_EQUAL | A_AND | A_OR ->
            invalid_arg "boolean node in Apron_domain.apron_of_expr"
        in
        e

    | T_float_const (l,h) -> 
        Texpr1.Cst (Coeff.Interval (Itv_float.to_apron (l,h)))

    | T_int_const (l,h) -> 
        Texpr1.Cst (Coeff.Interval (Itv_int.to_apron (l,h)))

    | T_bool_const False -> 
        Texpr1.Cst (Coeff.Interval (Itv_int.to_apron Itv_int.zero))

    | T_bool_const True ->
        Texpr1.Cst (Coeff.Interval (Itv_int.to_apron Itv_int.one))

    | T_bool_const Maybe ->
        Texpr1.Cst (Coeff.Interval (Itv_int.to_apron Itv_int.zero_one))

    | T_var v -> Texpr1.Var (apron_of_var v)


  let apron_of_cons env ((e,t,x):expr typed) (neg:bool) : Tcons1.earray =
    match e with
    | T_binary (op,e1,e2) ->
        let e1, e2 = apron_of_expr e1, apron_of_expr e2 in
        let op = if neg then cmp_neg op else op in
        let e1,op,e2 = match op with
        | A_EQUAL -> e1, Tcons1.EQ, e2
        | A_NOT_EQUAL -> e1, Tcons1.DISEQ, e2
        | A_LESS -> e2, Tcons1.SUP, e1
        | A_LESS_EQUAL -> e2, Tcons1.SUPEQ, e1
        | A_GREATER -> e1, Tcons1.SUP, e2
        | A_GREATER_EQUAL -> e1, Tcons1.SUPEQ, e2
        | _ -> invalid_arg "non-comparison node in Apron_domain.apron_of_cons"
        in
        let ar = Tcons1.array_make env 1 in        
        let e = Texpr1.Binop (Texpr1.Sub, e1, e2, Texpr1.Real, Texpr1.Near) in
        let c = Tcons1.make (Texpr1.of_expr env e) op in
        Tcons1.array_set ar 0 c;
        ar
    | _ -> invalid_arg "non-comparison node in Apron_domain.apron_of_cons"
          

  let fwd_assign a dst e =
    let ee = Texpr1.of_expr (Abstract1.env a) (apron_of_expr e) in
    let assign v = 
      Abstract1.assign_texpr man a (apron_of_var v) ee None
    in
    (match dst with
    | STRONG v -> assign v
    | WEAK vl ->
        List.fold_left (fun a v -> Abstract1.join man a (assign v)) a vl
    ), noerr

  let fwd_filter a e =
    let env = Abstract1.env a in
    let act, acf = apron_of_cons env e false, apron_of_cons env e true in
    Abstract1.meet_tcons_array man a act,
    Abstract1.meet_tcons_array man a acf,
    noerr
*)

  (**************************************************************************)





  (*************)
  (* UTILITIES *)
  (*************)


  (* apron name of a variable *)
  let apron_of_var (v:var) : Var.t =
    Var.of_string (string_of_id (v.var_name^"#") v.var_id)


  let add_var_to_env env v =
    let i,r = match v.var_typ with
    | A_int _ | A_BOOL -> [|apron_of_var v|], [||]
    | A_float _ -> [||], [|apron_of_var v|]
    in
    Environment.add env i r



  (* of Apron types *)
  (* ************** *)

  (* creates a linearization environment (variable to interval) *)
  let lin_env (a:t) : Lin.env =
    fun v -> Lin.I.of_apron (Abstract1.bound_variable man a (apron_of_var v))


  (* to Apron types *)
  (* ************** *)

  let apron_of_typ (t:typ) : Texpr1.typ * Texpr1.round =
    match t with
    | A_int _ | A_BOOL -> Texpr1.Int, Texpr1.Zero
    | A_float A_FLOAT -> Texpr1.Single, Texpr1.Near
    | A_float A_DOUBLE -> Texpr1.Double, Texpr1.Near
    | A_float A_REAL -> Texpr1.Real, Texpr1.Near


  let apron_of_lin env ((i,m):Lin.linexpr) : Linexpr1.t =
    let r = Linexpr1.make env in
    Linexpr1.set_cst r (Coeff.Interval (Lin.I.to_apron i));
    Lin.VMap.iter 
      (fun v i ->
        Linexpr1.set_coeff r 
          (apron_of_var v) (Coeff.Interval (Lin.I.to_apron i))
      )
      m;
    r

  let apron_of_cons_op = function 
    | Lin.L_EQ | Lin.L_EQ_INT -> Lincons1.EQ
    | Lin.L_NEQ | Lin.L_NEQ_INT -> Lincons1.DISEQ
    | Lin.L_GEQ | Lin.L_GEQ_INT -> Lincons1.SUPEQ
    | Lin.L_GT  -> Lincons1.SUP


   let apron_of_lincons env ((l,o):Lin.lincons) : Lincons1.t =
     Lincons1.make (apron_of_lin env l) (apron_of_cons_op o)
         

   let apron_of_lincons_ar env (c:Lin.lincons) : Lincons1.earray =
     let a = Lincons1.array_make env 1 in
     Lincons1.array_set a 0 (apron_of_lincons env c);
     a
    

  (* creates a linearization environment (variable to interval) *)
  let lin_env (a:t) : Lin.env =
    fun v -> Lin.I.of_apron (Abstract1.bound_variable man a (apron_of_var v))


  (* normalize equalities as pairs of inequalities *)
  let apron_of_lincons_array (ar:Lincons1.earray) : Lincons1.t list =
    let l = ref [] in
    for i=0 to Lincons1.array_length ar - 1 do
      let d = Lincons1.array_get ar i in
      if Lincons1.get_typ d = Lincons1.EQ then (
        Lincons1.set_typ d Lincons1.SUPEQ;
        let dd = Lincons1.copy d in
        Lincons1.set_cst dd (Coeff.neg (Lincons1.get_cst dd));
        Lincons1.iter (fun c v -> Lincons1.set_coeff dd v (Coeff.neg c)) dd;
        l := dd::(!l)
       );
      l := d::(!l)
    done;
    !l


  (* construct abstract element from constraint list *)
  let abs_of_lincons env (l:Lincons1.t list) : t =
    let ar = Lincons1.array_make env (List.length l) in
    let i = ref 0 in
    List.iter (fun c -> Lincons1.array_set ar !i c; incr i) l;
    Abstract1.of_lincons_array man env ar



  (* useful transfer functions *)
  (* ************************* *)

  (* bounds v within i *)
  let bound_var (a:t) (v:var) ((l,h):Lin.I.t) : t =
    if not (Lin.B.is_finite l || Lin.B.is_finite h) then a else
    let env = Abstract1.env a in
    let x,y = Lin.B.to_apron (Lin.B.neg l), Lin.B.to_apron h in
    let v = apron_of_var v in
    let ar = Lincons1.array_make env 2 in
    if Lin.B.is_finite l then (
      let c = Lincons1.make( Linexpr1.make env) Lincons1.SUPEQ in
      Lincons1.set_list c [Coeff.s_of_int 1, v] (Some (Coeff.Scalar x));
      Lincons1.array_set ar 0 c
     );
    if Lin.B.is_finite h then (
      let c = Lincons1.make( Linexpr1.make env) Lincons1.SUPEQ in
      Lincons1.set_list c [Coeff.s_of_int (-1), v] (Some (Coeff.Scalar y));
      Lincons1.array_set ar 1 c
     );
    Abstract1.meet_lincons_array man a ar

  (* assigns the interval i to v *)
  let set_var_itv (a:t) (v:var) (i:Lin.I.t) : t =
    bound_var (Abstract1.forget_array man a [|apron_of_var v|] false) v i

  (* adds a constant interval i to v (c must be bounded) *)
  let shift_var_itv (a:t) (v:var) (i:Lin.I.t) : t =
    assert(Lin.B.is_finite (fst i) && Lin.B.is_finite (snd i));
    if Lin.I.equal i Lin.I.zero then a else
    let i = Lin.I.to_apron i in
    let v = apron_of_var v in
    let e = Linexpr1.make (Abstract1.env a) in
    Linexpr1.set_list e [Coeff.s_of_int 1, v] (Some (Coeff.Interval i));
    Abstract1.assign_linexpr man a v e None


  (* backward version of shift_var *)
  let bwd_shift_var_itv (a:t) (v:var) ((l,h) as i:Lin.I.t) : t =
    if Lin.I.equal i Lin.I.zero then a else
    (* shift v by -l and -h, then intersect *)
    (* all operations must be under-approximated (i.e., exact!) *)
    let sl,sh = Lin.B.neg l, Lin.B.neg h in
    let a1,a2 = shift_var_itv a v (sl,sl), shift_var_itv a v (sh,sh) in
    Abstract1.meet man a1 a2


  (* backward version of V := [-oo;+oo] *)
  let bwd_forget_var (a:t) (v:var) : t =
    if Abstract1.is_variable_unconstrained man a (apron_of_var v)
    then a else Abstract1.bottom man (Abstract1.env a)

 
  (* backward version of set_var *)
  let bwd_set_var_itv (a:t) (v:var) ((l,h) as i:Lin.I.t) : t =
    (* all operations must be under-approximated (i.e., exact!) *)
    match Lin.B.is_finite l, Lin.B.is_finite h with
    | true, true -> 
        (* bound v in i *)
        let a = bound_var a v i in
        (* shift *)
        let a = bwd_shift_var_itv a v i in
        (* forget v *)
        Abstract1.forget_array man a [|apron_of_var v|] false
    | _ ->
        (* TODO: more precise handling of half-infinite intervals *)
        bwd_forget_var a v

    
  (* backward version of the copy assignment v:=w *)
  let bwd_copy_var (a:t) (v:var) (w:var) : t =
    (* all operations must be under-approximated (i.e., exact!) *)
    let v,w = apron_of_var v, apron_of_var w in
    (* adds the constaint V=W *)
    let ar = Lincons1.array_make (Abstract1.env a) 1 in
    let c = Lincons1.array_get ar 0 in
    Lincons1.set_list c [Coeff.s_of_int 1, v; Coeff.s_of_int (-1), w] None;
    let a = Abstract1.meet_lincons_array man a ar in
    (* forget v *)
    Abstract1.forget_array man a [|v|] false


  let trace_remove = false

  (* physically remove l from a, if it occurs *)
  let remove_constraint (a:t) (l:Lin.lincons) : t =
    let c = apron_of_lincons (Abstract1.env a) l in
    let l = apron_of_lincons_array (Abstract1.to_lincons_array man a) in
    let ll = List.filter (fun cc -> c <> cc) l in
    if List.length l <> List.length ll then (
      if trace_remove then 
        Format.printf "### remove_constraint remove: %a ###@\n" Lincons1.print c;
      abs_of_lincons (Abstract1.env a) ll 
     )
    else a

(*
  (* try to replacea constraint from a with c, while staying equivalent *)
  let replace_equiv (a:t) (c:Lincons1.t) : Lincons1.t list =
    let org = abs_to_lincons a in

    List.fold_left
      (fun l ->
      )
*)

  (* as remove_constraint, but first replaces constraint in a with equivalent
     constrints from pool
   *)
  let remove_constraint_pool (a:t) (pool:t) (l:Lin.lincons) : t =
     a




  (********************)
  (* GRAPHICAL OUTPUT *)
  (********************)


  let svg_abstract (ch:out_channel) (a:t) ((x,y):var*var) =
    let w,h,x0,y0,x1,y1 = 500,500, -130.,-130., 130.,130. in
    let xx,yy = apron_of_var x, apron_of_var y in
    (* extract linexpr coeff *)
    let of_coeff = function
      | Coeff.Scalar s -> Float.Double.of_apron_up s
      | Coeff.Interval i -> fst (Itv_float.of_apron i)
    in
    let get_coord l = 
      of_coeff (Linexpr1.get_cst l),
      of_coeff (Linexpr1.get_coeff l xx), 
      of_coeff (Linexpr1.get_coeff l yy)
    in
    (* project on (x,y) *)
    let env = add_var_to_env (add_var_to_env (Environment.make [||] [||]) x) y 
    in
    let a = Abstract1.change_environment man a env true in
    (* get & convert constraints *)
    let lin = Abstract1.to_lincons_array man a in
    let l = 
      Array.init
        (Lincons1.array_length lin)
        (fun i ->
          let v = Lincons1.array_get lin i in
          get_coord (Lincons1.get_linexpr1 v), Lincons1.get_typ v
        )
    in
    (* bound *)
    let b = bound_var a x (Lin.I.of_floats x0 x1) in
    let b = bound_var b y (Lin.I.of_floats y0 y1) in
    if Abstract1.is_bottom man b then () else
    (* get & convert generators *)
    let gen = Abstract1.to_generator_array man b in
    let g = 
      Array.to_list 
        (Array.init
           (Generator1.array_length gen)
           (fun i -> 
             get_coord (Generator1.get_linexpr1 (Generator1.array_get gen i))
        ))
    in
    (* sort generators *)
    let l_to_r = 
      List.sort
        (fun (_,x,y) (_,x',y') -> if x=x' then compare y' y else compare x x')
        g 
    in
    let (_,_,u) = List.hd l_to_r in
    let up = List.filter (fun (_,_,y) -> y>=u) l_to_r
    and down = List.rev (List.filter (fun (_,_,y) -> y<u) l_to_r)
    in
    (* coordinate transform *)
    let xx x = (float_of_int w) *. (x-.x0) /. (x1-.x0)
    and yy y = (float_of_int h) *. (y-.y1) /. (y0-.y1)
    in
    (* output header *)
    Printf.fprintf ch "<?xml version=\"1.0\" standalone=\"no\"?>\n<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\" \"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\n";
    Printf.fprintf ch "<svg width=\"%ipx\" height=\"%ipx\">\n" w h;
    (* output background *)
    Printf.fprintf ch "<rect fill=\"white\" stroke=\"none\" x=\"%i\" y=\"%i\" width=\"%i\" height=\"%i\"/>\n" 0 0 w h;
    (* output polygon *)
    Printf.fprintf ch "<polygon fill=\"lime\" points=\"";
    List.iter (fun (_,x,y) -> Printf.fprintf ch "%g,%g " (xx x) (yy y)) (up@down);
    Printf.fprintf ch "\"/>\n";
    (* output constraints *)
    Array.iter
      (fun ((c,a,b),op) ->
        let p = [(-.c-.b*.y1)/.a, y1;  x1, (-.c-.a*.x1)/.b;
                 (-.c-.b*.y0)/.a, y0;  x0, (-.c-.a*.x0)/.b]
        in
        (* constraint line *)
        let p = List.sort (fun a b -> compare (fst a) (fst b)) p in
        let p = List.filter (fun (x,y) -> x>=x0 && x<=x1 && y>=y0 && y<=y1) p in
        Printf.fprintf ch "<polyline points=\"";
        List.iter (fun (x,y) -> Printf.fprintf ch "%g,%g " (xx x) (yy y)) p;
        Printf.fprintf ch "\" stroke=\"blue\" stroke-width=\"1px\" fill=\"none\"/>\n";
        (* constraint test *)
        if List.length p > 0 then
          let u,v = List.hd p in
          let u = 0.95*.u+.0.05*.(x0+.x1)/.2. in
          let v = 0.95*.v+.0.05*.(y0+.y1)/.2. in
          let ll = Lin.A.set_var x (Lin.I.of_float a) (Lin.A.set_var y (Lin.I.of_float b) (Lin.A.cst (Lin.I.of_float c))) in
          Printf.fprintf ch "<text x=\"%g\" y=\"%g\" fill=\"midnightblue\" font-size=\"16px\" font-family=\"helvetica\" text-anchor=\"%s\">%s %s 0</text>\n" (xx u) (yy v) (if u < (x0+.x1)/.2. then "start" else "end") (Lin.A.to_string ll) (Lincons1.string_of_typ op)
      )
      l;
    (* output coordinate system *)
    Printf.fprintf ch "<line x1=\"%i\" y1=\"%f\" x2=\"%i\" y2=\"%g\" stroke=\"black\" stroke-width=\"1px\"/>\n" 0 (yy 0.) w (yy 0.);
    Printf.fprintf ch "<line x1=\"%g\" y1=\"%i\" x2=\"%g\" y2=\"%i\" stroke=\"black\" stroke-width=\"1px\"/>\n" (xx 0.) 0 (yy 0.) h;
    Printf.fprintf ch "<text x=\"%i\" y=\"%g\" fill=\"black\" font-size=\"16px\" font-family=\"helvetica\" text-anchor=\"end\">%s</text>\n" (w-8) ((yy 0.)-.6.) x.var_name;
    Printf.fprintf ch "<text x=\"%g\" y=\"%i\" fill=\"black\" font-size=\"16px\" font-family=\"helvetica\" text-anchor=\"start\">%s</text>\n" ((xx 0.)+.6.) 20 y.var_name;

    (* output generator coord *)
    List.iter (fun (_,x,y) -> Printf.fprintf ch "<text x=\"%g\" y=\"%g\" fill=\"darkgreen\" font-size=\"16px\" font-family=\"helvetica\" text-anchor=\"middle\">(%g,%g)</text>\n" (xx x) (if y >= u then yy y -. 16. else yy y +. 16.) x y) (up@down);
    (* output footer *)
    Printf.fprintf ch "</svg>\n"


  let print_vars nm a vars =
    match vars with
    | [x;y] ->
        let ch = open_out nm in
        svg_abstract ch a (x,y);
        close_out ch
    | _ -> ()



  (* ********************* *)
  (* CREATION & INSPECTION *)
  (* ********************* *)


  let empty () =
    Abstract1.top man (Environment.make [||] [||])

  let bot a =
    Abstract1.bottom man (Abstract1.env a)

  let top a =
    Abstract1.top man (Abstract1.env a)

  let is_bot a =
    Abstract1.is_bottom man a
      
  let subseteq a b =
    Abstract1.is_leq man a b
      
  let print fmt a =
    Abstract1.print fmt a
      


  (* ***************** *)
  (* FORWARD OPERATORS *)
  (* ***************** *)


  let fwd_join a b m =
    match m with
    | MERGE -> Abstract1.join man a b
    | WIDEN -> Abstract1.widening man a b


  let fwd_add_var a v =
    let env = add_var_to_env (Abstract1.env a) v in
    let a = Abstract1.change_environment man a env false in
    bound_var a v (Lin.type_itv v.var_typ)

  let fwd_del_var a v =
    let env = Environment.remove (Abstract1.env a) [|apron_of_var v|] in
    Abstract1.change_environment man a env false


  let fwd_assign a dst e =
    let l = Lin.linearize_expr true (lin_env a) e in
    let ee = apron_of_lin (Abstract1.env a) l in
    let assign v = Abstract1.assign_linexpr man a (apron_of_var v) ee None
    in
    (match dst with
    | STRONG v ->
        if v.var_scope = T_VOLATILE || v.var_scope = T_INPUT
        then set_var_itv a v (Lin.A.eval (lin_env a) l)
        else assign v
    | WEAK vl ->
        List.fold_left (fun a v -> Abstract1.join man a (assign v)) a vl
    ), noerr


  let fwd_filter a e =
    let env = Abstract1.env a in
    let c1 = Lin.linearize_cons true (lin_env a) e in
    let c2 = Lin.neg_cons c1 in
    Abstract1.meet_lincons_array man a (apron_of_lincons_ar env c1),
    Abstract1.meet_lincons_array man a (apron_of_lincons_ar env c2),
    noerr



  (* ****************** *)
  (* BACKWARD OPERATORS *)
  (* ****************** *)

  let trace_bwd = false

  let bwd_trace msg f x =
    let r = f x in
    if trace_bwd then Format.printf "**** %s => %a@\n" msg print r;
    r


  let bwd_add_var post v pre =
    let (l,h) = Lin.type_itv v.var_typ in
    if trace_bwd then Format.printf "** bwd_add_var %s %s@\n" v.var_name (Lin.I.to_string (l,h));
    let post = bwd_trace "bwd_set_var_itv" (bwd_set_var_itv post v) (l,h) in
    let post = bwd_trace "fwd_del_var" (fwd_del_var post) v in
    bwd_trace "meet" (Abstract1.meet man pre) post
      

  let bwd_del_var post v pre =
    if trace_bwd then Format.printf "** bwd_del_var %s@\n" v.var_name;
    let post = bwd_trace "fwd_add_var" (fwd_add_var post) v in
    bwd_trace "meet" (Abstract1.meet man pre) post


  let bwd_meet a b m =
    match m with
    | MERGE -> Abstract1.meet man a b
    | WIDEN ->
        if subseteq a b then a else (
        Format.printf "arg1 = %a@\narg2 = %a@\n" print a print b;
        bot a
       )


  (* helper for bwd_filter and non-invertible bwd_assign *)
  let add_space post c =
    let env = Abstract1.env post in
    if trace_bwd then Format.printf "**** add_space %s@\n" (Lin.cons_to_string c);
    let post = bwd_trace "** meet_constraint" 
        (Abstract1.meet_lincons_array man post) (apron_of_lincons_ar env c)
    in
    if is_bot post then
      (* build element with the negation of c *)
      let c = apron_of_lincons_ar env (Lin.neg_cons c) in
      bwd_trace "** of_lincons" (Abstract1.of_lincons_array man env) c
    else
      (* remove constraint *)
      bwd_trace "** remove_constraint" (remove_constraint post) c
        

  let bwd_filter post1 post2 err e pre =
    let do_constraint post ((((l,h),m),op) as c) =
      match op with
      | Lin.L_GT | Lin.L_GEQ | Lin.L_GEQ_INT ->
          (* aff + [l;h] >= 0 => aff + h >= 0 *)
          let post = add_space post (((h,h),m),op) in
          bwd_trace "meet_pre" (Abstract1.meet man pre) post
      | Lin.L_EQ | Lin.L_EQ_INT ->
          (* aff + [l;h] = 0 => aff + h >= 0 \/ aff + l <= 0 *)
          let (((_,h1),m1),op1), (((_,h2),m2),op2) = Lin.split_cons c in
          let post = add_space post (((h1,h1),m1),op1) in
          let post = add_space post (((h2,h2),m2),op2) in
          bwd_trace "meet_pre" (Abstract1.meet man pre) post
      | Lin.L_NEQ | Lin.L_NEQ_INT -> 
          if not (Lin.B.equal l h) then Abstract1.meet man pre post else
          (* aff + c = 0 => aff + c <= 0 /\ aff + c >= 0 *)
          let c1,c2 = Lin.split_cons c in
          let post1 = add_space post c1 in
          let post2 = add_space post c2 in
          let post = bwd_trace "meet_neq" (Abstract1.meet man post1) post2 in
          bwd_trace "meet_pre" (Abstract1.meet man pre) post
    in
    let c1 = Lin.linearize_cons true (lin_env pre) e in
    let c2 = Lin.neg_cons c1 in
    if trace_bwd then Format.printf "** bwd_filter true %s@\n" (Lin.cons_to_string c1);
    let mid1 = do_constraint post1 c1 in
    if trace_bwd then Format.printf "** bwd_filter false %s@\n" (Lin.cons_to_string c2);
    let mid2 = do_constraint post2 c2 in
    bwd_trace "meet_both" (Abstract1.meet man mid1) mid2


  (* improvement on bwd_filter: if it fails, tries to force a branch *)
  let bwd_filter post1 post2 err e pre =
    (* try both branches *)
    let r = bwd_filter post1 post2 err e pre in
    if not (is_bot r) then r else
    (* try to force the true branch *)  (
    if trace_bwd then Format.printf "** bwd_filter force true branch@\n";
    let r = bwd_filter post1 (bot post2) err e pre in
    if not (is_bot r) then r else (
    (* try to force the false branch *)
    if trace_bwd then Format.printf "** bwd_filter force false branch@\n";
    bwd_filter (bot post1) post2 err e pre))


  (* special version of add_space that tries to maximize v *)
  let add_space_var post v l =
    let env = Abstract1.env post in
    (* get the equality section *)
    let eq = apron_of_lincons_ar env (l,Lin.L_EQ) in
    let sec = Abstract1.meet_lincons_array man post eq in
    (* extrude along v *)
    let prism = bwd_trace "prism" (Abstract1.forget_array man sec [|apron_of_var v|]) false in
    (* meet with post *)
    let post = bwd_trace "meet_prism" (Abstract1.meet man post) prism in
    (* finally, add space *)
    add_space post (l,Lin.L_GEQ)


  let bwd_assign post err dst e pre =
    let env = Abstract1.env pre in
    let i,m = Lin.linearize_expr true (lin_env pre) e in
    let post = match dst with 
    | WEAK _ -> failwith "bwd_assign WEAK: TODO"
    | STRONG v ->
        if trace_bwd then Format.printf "** bwd_assign %s := %s@\n" v.var_name (Lin.expr_to_string (i,m));
        if Lin.VMap.is_empty m then
          (* constant case *)
          bwd_trace "bwd_set_var" (bwd_set_var_itv post v) i
        else if Lin.VMap.mem v m then
          (* invertible case *)
          let post = bwd_trace "bwd_shift_var" (bwd_shift_var_itv post v) i in
          let l = Lin.invert true (lin_env pre) (Lin.I.zero,m) v in
          if trace_bwd then Format.printf "**** inverted %s@\n" (Lin.expr_to_string l);
          let e = apron_of_lin env l in
          bwd_trace "assign_linexpr" (Abstract1.assign_linexpr man post (apron_of_var v) e) None
        else
          (* non-invertible case: handled as forget then add equality *)
          (* backward equality *)
          let ((l,h),m) = Lin.A.set_var v Lin.I.minus_one (i,m) in
          let post = add_space_var post v ((h,h),m) in
          let post = add_space_var post v (Lin.A.neg ((l,l),m)) in
          (* backward forget *)
          bwd_trace "bwd_forget_var" (bwd_forget_var post) v
          
    in
    bwd_trace "meet" (Abstract1.meet man pre) post



end: ABSTRACT_DOMAIN)



(* INSTANTIATIONS *)
(* ************** *)

module PolkaDomain =
  ApronDomain
    (struct
      type lib = Polka.strict Polka.t
      let manager = Polka.manager_alloc_strict ()
    end)

module OctagonDomain =
  ApronDomain
    (struct
      type lib = Oct.t
      let manager = Oct.manager_alloc ()
    end)

