(*   
   Forward / backward iterator for Typed_syntax programs, in a Domain
   abstract domain.

   Copyright (C) 2011 Antoine Miné
*)

open Datatypes
open Abstract_syntax
open Typed_syntax
open Domain

let log = ref Format.std_formatter
let trace_lfp = ref false
let trace_gfp = ref false
let trace_fwd = ref false
let trace_bwd = ref false
let trace_iter = ref false
let dump_inv = ref true
let enable_print = ref true

let nb_unroll = ref 2
let nb_join = ref 2
let nb_down = ref 2
let nb_meet = ref 4

let latex_out = ref None


(************************************************************************)
(* FLOW ABSTRACT DOMAIN *)
(************************************************************************)



(* this functor adds continuation abstract environements, which is useful to
   handle non-structured constructions (return, break, forward gotos)
 *)
module Flow(D:ABSTRACT_DOMAIN) = (struct

  type t = {
      (* current environment *)
      env: D.t;

      (* continuations *)
      conts: D.t StringMap.t;
    }


  let empty () = 
    { env = D.empty (); conts = StringMap.empty; }


  (* utilities *)

  (* keep elements appearing in only one map *)
  let map_join f x y =
    StringMap.map2zo (fun _ x -> x) (fun _ x -> x) (fun _ x y -> f x y) x y

  (* discard elements appearing in only one map *)
  let map_meet f x y =
    StringMap.fold2zo
      (fun s _ a -> StringMap.remove s a) (fun s _ a -> a)
      (fun s x y a -> StringMap.add s (f x y) a)
      x y x

  (* used to merge continuations in both arguments *)
  let bwd_map_meet x y = 
    map_meet (fun x y -> D.bwd_meet x y MERGE) x y
  
  (* applies f to all common flows *)
  let bwd_map f post pre =
    { env = f post.env pre.env;
      conts = map_meet f post.conts pre.conts;
    }

  (* applies f to all flows *)
  let fwd_map f x =
    { env = f x.env;
      conts = StringMap.map (fun e -> f e) x.conts;
    }


  (* these operators only operate on current environments *)

  let bot x =
    { x with env = D.bot x.env; }

  let top x =
    { x with env = D.top x.env; }

  let is_bot x =
    D.is_bot x.env

  let print fmt x =
    D.print fmt x.env;
    StringMap.iter (fun s e -> Format.fprintf fmt ";@ %s:%a" s D.print e) x.conts

  let print_svg o ?color ?window size x vars = 
    D.print_svg o ?color ?window size x.env vars

  let print_html out x = D.print_html out x.env
  let print_latex out x = D.print_latex out x.env

  let subseteq a b =
    (D.subseteq a.env b.env) &&
    (StringMap.for_all2o 
       (fun _ _ -> false) (fun _ _ -> true) (fun _ -> D.subseteq) 
       a.conts b.conts)

  let fwd_assign x dst e =
    let env, err = D.fwd_assign x.env dst e in
    { x with env = env; }, err

  let fwd_filter x e =
    let env1,env2,err = D.fwd_filter x.env e in
    { x with env = env1; }, { x with env = env2; }, err


  let bwd_assign post err dst e pre =
    { env = D.bwd_assign post.env err dst e pre.env; 
      conts = bwd_map_meet post.conts pre.conts;
    }

  let bwd_filter post1 post2 err e typ pre =
    { env = D.bwd_filter post1.env post2.env err e typ pre.env; 
      conts = bwd_map_meet post1.conts (bwd_map_meet post2.conts pre.conts);
    }


  (* these operators operate on environements and continuations *)

  let fwd_add_var x v = fwd_map (fun e -> D.fwd_add_var e v) x
  let fwd_del_var x v = fwd_map (fun e -> D.fwd_del_var e v) x

  let fwd_join x y t =
    { env = D.fwd_join x.env y.env t; 
      conts = map_join (fun x y -> D.fwd_join x y t) x.conts y.conts;
    }

  let bwd_add_var post v pre = 
    bwd_map (fun x y -> D.bwd_add_var x v y) post pre

  let bwd_del_var post v pre = 
    bwd_map (fun x y -> D.bwd_del_var x v y) post pre

  let bwd_meet x y t =
    bwd_map (fun x y -> D.bwd_meet x y t) x y



  (* continuation management *)

  let get_conts x =
    { x with conts = StringMap.empty; }, x.conts

  let put_conts x conts =
    if not (StringMap.is_empty x.conts) then 
      failwith "non-empty continuation (backward goto?)";
    { x with conts = conts; }

  let fwd_goto x lbl =
    let r = 
      try D.fwd_join x.env (StringMap.find lbl x.conts) MERGE
      with Not_found -> x.env
    in
    { env = D.bot x.env;
      conts = StringMap.add lbl r x.conts; 
    }

  let fwd_label x lbl =
    let r = 
      try D.fwd_join x.env (StringMap.find lbl x.conts) MERGE
      with Not_found -> x.env
    in
    { env = r; 
      conts = StringMap.remove lbl x.conts;
    }


  let bwd_goto post lbl pre =
    let r = 
      try D.bwd_meet pre.env (StringMap.find lbl post.conts) MERGE
      with Not_found -> D.bot pre.env
    in
    { env = r;
      conts = bwd_map_meet post.conts pre.conts;
    }

  let bwd_label post lbl pre =
    let conts = StringMap.add lbl post.env post.conts in
    { env = D.bwd_meet pre.env post.env MERGE;
      conts = bwd_map_meet pre.conts conts;
    }

end)



(************************************************************************)
(* BOOLEAN EXPRESSION ABSTRACTION *)
(************************************************************************)



(* this functor handles the boolean part of expressions *)
module Boolean(D:ABSTRACT_DOMAIN) = (struct
  
  include D


  (* forward *)
  (* ******* *)

  (* calls D.fwd_filter on expressions with a comparison root and only
     numerical internal node
   *)
  let rec fwd_filter a (e,t,x) =
    match e with

    | T_unary (A_NOT, e1) ->
        let a1,a2,err = fwd_filter a e1 in a2,a1,err

    | T_unary (A_cast (A_BOOL,_), ((_,t,_) as e1)) ->
        if t = A_BOOL then fwd_filter a e1 else
        (* x <=> x != 0 *)
        fwd_filter a (T_binary (A_NOT_EQUAL, e1,
                                (T_int_const Itv_int.zero,t,x)), t, x)

    | T_binary (A_EQUAL, ((_,t,_) as e1), e2) ->
        if t <> A_BOOL then D.fwd_filter a (e,t,x) else
        (* x = y <=> (x and y) or (!x and !y) *)
        let a1,a2,err1 = fwd_filter a e1 in
        let a11,a12,err2 = fwd_filter a1 e2
        and a21,a22,err3 = fwd_filter a2 e2 in
        D.fwd_join a11 a22 MERGE, D.fwd_join a12 a21 MERGE,
        join_err (join_err err1 err2) err3
          
    | T_binary (A_NOT_EQUAL, ((_,t,_) as e1), e2) ->
        if t <> A_BOOL then D.fwd_filter a (e,t,x) else
        let a1,a2,err = fwd_filter a (T_binary (A_EQUAL, e1, e2), t, x) in
        a2,a1,err

    | T_binary ((A_LESS | A_GREATER | A_LESS_EQUAL | A_GREATER_EQUAL), _, _) ->
        D.fwd_filter a (e,t,x)

    | T_binary (A_AND, e1, e2) -> 
        let a1,a2,err1 = fwd_filter a e1 in
        let a11,a12,err2 = fwd_filter a1 e2 in
        a11, D.fwd_join a2 a12 MERGE, join_err err1 err2

    | T_binary (A_OR, e1, e2) -> 
        let a1,a2,err1 = fwd_filter a e1 in
        let a21,a22,err2 = fwd_filter a2 e2 in
        D.fwd_join a1 a21 MERGE, a22, join_err err1 err2

      (* boolean x <=> x != 0 *)
    | T_var v when v.var_typ = A_BOOL -> 
        let t = A_int (A_INTEGER, A_SIGNED) in
        fwd_filter a (T_binary (A_NOT_EQUAL, (e,t,x), 
                                (T_int_const Itv_int.zero,t,x)), t, x)

    | T_bool_const True -> a, D.bot a, noerr
    | T_bool_const False -> D.bot a, a, noerr
    | T_bool_const Maybe -> a, a, noerr

    | _ -> invalid_arg "non-boolean expression in Boolean.fwd_filter"


  (* calls D.fwd_assign with only numerical nodes *)
  let fwd_assign a dst ((_,t,x) as e) =
    if t <> A_BOOL then D.fwd_assign a dst e else
    (* b = expr <=> if expr then b=1 else b=0 *)
    let a1,a2,err = fwd_filter a e in
    let a1,err1 = D.fwd_assign a1 dst (T_int_const Itv_int.one, t, x) 
    and a2,err2 = D.fwd_assign a2 dst (T_int_const Itv_int.zero, t, x) in
    D.fwd_join a1 a2 MERGE, join_err err (join_err err1 err2)



  (* backward *)
  (* ******** *)

  (* when an operator is decomposed into sequences of simpler ones
     (and, or, etc.), we need to reconstuct the intermediate invariants
     by calling some fwd_ functions
   *)

  let rec bwd_filter post1 post2 err (e,t,x) typ pre =
    match e with

    | T_unary (A_NOT, e1) ->
        bwd_filter post2 post1 err e1 typ pre

    | T_unary (A_cast (A_BOOL,_), ((_,t,_) as e1)) ->
        if t = A_BOOL then bwd_filter post1 post2 err e1 typ pre else
        (* x <=> x != 0 *)
        bwd_filter post1 post2 err
          (T_binary (A_NOT_EQUAL, e1, (T_int_const Itv_int.zero,t,x)), t, x)
          typ pre

    | T_binary (A_EQUAL, ((_,t,_) as e1), e2) ->
        if t <> A_BOOL then D.bwd_filter post1 post2 err (e,t,x) typ pre else
        (* x = y <=> (x and y) or (!x and !y) *)
        (* reconstruct the forward effect of e1 alone *)
        let mid1,mid2,_ = fwd_filter pre e1 in
        (* backward e2 *)
        let mid1 = bwd_filter post1 post2 err e2 typ mid1 in
        let mid2 = bwd_filter post1 post2 err e2 typ mid2 in
        (* backward e1 *)
        bwd_filter mid1 mid2 err e1 typ pre

    | T_binary (A_NOT_EQUAL, ((_,t,_) as e1), e2) ->
        if t <> A_BOOL then D.bwd_filter post1 post2 err (e,t,x) typ pre else
        bwd_filter post2 post1 err (T_binary (A_EQUAL, e1, e2), t, x) typ pre

    | T_binary ((A_LESS | A_GREATER | A_LESS_EQUAL | A_GREATER_EQUAL), _, _) ->
        D.bwd_filter post1 post2 err (e,t,x) typ pre

    | T_binary (A_AND, e1, e2) -> 
        (* reconstruct the forward effect of e1 alone *)
        let mid1,_,_ = fwd_filter pre e1 in
        (* backward e2 post -> mid *)
        let mid1 = bwd_filter post1 post2 err e2 typ mid1 in
        (* backward e1 mid -> pre *)
        bwd_filter mid1 post2 err e1 typ pre

    | T_binary (A_OR, e1, e2) -> 
        (* reconstruct the forward effect of e1 alone *)
        let _,mid2,_ = fwd_filter pre e1 in
        (* backward e2 post -> mid *)
        let mid2 = bwd_filter post1 post2 err e2 typ mid2 in
        (* backward e1 mid -> pre *)
        bwd_filter post1 mid2 err e1 typ pre

      (* boolean x <=> x != 0 *)
    | T_var v when v.var_typ = A_BOOL -> 
        let t = A_int (A_INTEGER, A_SIGNED) in
        bwd_filter post1 post2 err
          (T_binary (A_NOT_EQUAL, (e,t,x), 
                     (T_int_const Itv_int.zero,t,x)), t, x)
          typ pre

    | T_bool_const True ->
        D.bwd_meet pre post1 MERGE

    | T_bool_const False ->
        D.bwd_meet pre post2 MERGE

    | T_bool_const Maybe -> 
        D.bwd_meet pre (D.bwd_meet post1 post2 MERGE) MERGE

    | _ -> invalid_arg "non-boolean expression in Boolean.bwd_filter"


  let bwd_assign post err dst ((_,t,x) as e) pre =
    if t <> A_BOOL then D.bwd_assign post err dst e pre else
    (* b = expr <=> if expr then b=1 else b=0 *)
    (* reconstruct the forward effect of e *)
    let mid1,mid2,_ = fwd_filter pre e in
    (* backward assign of 0 / 1 *)
    let mid1 = D.bwd_assign post err dst (T_int_const Itv_int.one,t,x) mid1 in
    let mid2 = D.bwd_assign post err dst (T_int_const Itv_int.zero,t,x) mid2 in
    (* backward branching on e *)
    bwd_filter mid1 mid2 err e IFTHENELSE pre


end: ABSTRACT_DOMAIN)



(************************************************************************)
(* ITERATOR *)
(************************************************************************)



module Iterator(D:ABSTRACT_DOMAIN) = struct

  module F = Flow(Boolean(D))

  type loop =
    | Unroll of int
    | After of int

  type ctx = 
    | Call of string * id
    | Loop of label * loop

  type stack = ctx list (* call stack, top element first *)

  let string_of_ctx = function
    | Call (f,_) -> f
    | Loop (e,Unroll i) -> Printf.sprintf "loop=%i" i
    | Loop (e,After i) -> Printf.sprintf "loop>%i" i

  let rec print_stack fmt = function
    | [] -> ()
    | [a] -> Format.fprintf fmt "%s" (string_of_ctx a)
    | a::b -> Format.fprintf fmt "%s:%a" (string_of_ctx a) print_stack b


  let print_point fmt (((id,pos),stack):label*stack) =
    Format.fprintf fmt "{%s}:%s [%a]" 
      (string_of_id "" id) (string_of_position pos) 
      print_stack (List.rev stack)

  let print_extent fmt x =
    Format.fprintf fmt "%s" (string_of_extent x)
      
  let call_ctx f = Call (f.func_name, f.func_id)


  let print_svg_file nm x vars =
    match vars with
    | [a;b] ->
        let f = open_out nm in
        output_string f "<?xml version=\"1.0\" standalone=\"no\"?>\n<!DOCTYPE svg PUBLIC \"-//W3C//DTD SVG 1.1//EN\" \"http://www.w3.org/Graphics/SVG/1.1/DTD/svg11.dtd\">\n";
        F.print_svg f ~color:false (*~window:(-150.,-150.,150.,150.)*) (320,320) x (a,b);
        close_out f
    | _ -> ()
  
  let print_latex_file nm x =
    match !latex_out with
    | None -> ()
    | Some f -> Printf.fprintf f "%% %s\n$%a$\n\n" nm F.print_latex x


  (* utility: select vars for printing *)
  let all_vars = ref IdMap.empty
  let choose_two_vars () = 
    IdMap.fold
      (fun _ v acc-> 
        if List.length acc < 2 && not v.var_synthetic
        then v::acc else acc
      ) !all_vars []
  let graph_debug s c =
    print_svg_file ("out/"^s^".svg") c (choose_two_vars ());
    print_latex_file s c



  let rec goto_return = function
    | (Call (_,id))::_ -> return_label id
    | _::b -> goto_return b
    | [] -> failwith "return with no enclosing function"
          
  let goto_break = function
    | (Loop (lbl,_))::_ -> break_label lbl
    | _ -> failwith "break with no enclosing loop"
  


  (* invariant map *)
  (* ************* *)

  module InvKey =
    struct
      type t = label * stack 
      let compare ((id1,pos1),stack1) ((id2,pos2),stack2) =
        let x = compare_position pos1 pos2 in
        if x <> 0 then x else compare (id1,stack1) (id2,stack2)
    end

  module InvMap = Mapext.Make(InvKey)

  let fwd_inv_map = ref InvMap.empty
  let bwd_inv_map = ref InvMap.empty

  (* extra info for trace iter *)
  let iter = ref ""

  let add_inv_fwd lbl st a = 
    if !trace_iter then
      graph_debug (string_of_id ("fwd_iter_" ^ !iter) (fst lbl)) a;
    fwd_inv_map := InvMap.add (lbl,st) a !fwd_inv_map

  let add_inv_bwd lbl st a = 
    if !trace_iter then
      graph_debug (string_of_id ("bwd_iter_" ^ !iter) (fst lbl)) a;
    bwd_inv_map := InvMap.add (lbl,st) a !bwd_inv_map
 
  let get_inv_fwd lbl st = 
    try InvMap.find (lbl,st) !fwd_inv_map
    with Not_found ->
      Format.fprintf !log "invariant for %a Not_found@\n" print_point (lbl,st);
      raise Not_found

  let print_inv pref map rev fmt =
    let l = InvMap.fold (fun p a acc -> (p,a)::acc) map [] in
    let l = if rev then l else List.rev l in
    List.iter
      (fun (p,a) -> 
        (* do not print global initialization *)
        if snd p <> [] then (
          Format.fprintf fmt "%a:@\n%a@\n@\n" print_point p F.print a;
          let (id,pos),_ = p in
          graph_debug (string_of_id (pref^"_inv_") id) a
         )
      ) 
      l

  let print_inv_fwd ch = print_inv "fwd" !fwd_inv_map false ch
  let print_inv_bwd ch = print_inv "bwd" !bwd_inv_map true ch



  (* fix-points *)
  (* ********** *)

  (* over-approximation of body's smallest fixpoint greater than init:
     1) unroll some iterations
     2) then accumulate the next iterations with a join
     3) then accumulate with a widening until stabilisation
     4) perform some decreasing iterations
     return 1 U 4
   *)


  (* loop label:
     ----------

                   v
                   |
                   |init
      post         |
     ----------   /
     |         \ /
     |          |
     |          |inv
     ^          |
     |         / \
     |        /   \
     |       |     |
     |    pre|     |
     |       |     |
     |      body   |
     |       |     |
     |       |     |exit
     ---------     |
                   v

   *)

  let fwd_lfp ctx lbl body init =
    let old_iter = !iter in
    let n = ref 1 in

    (* unrolling *)
    let rec unroll acc init nb = 
      iter := Printf.sprintf "%s%02i_unroll_" old_iter !n; incr n;
      if nb > !nb_unroll then F.fwd_join acc (iter_up init init 1) MERGE else
      let ctx = (Loop (lbl,Unroll nb))::ctx in
      add_inv_fwd lbl ctx init;
      let post = body ctx init in
      if !trace_lfp then
        Format.fprintf !log "lfp:unrolling %i / %i@\n:%a@\n" nb !nb_unroll F.print post;
      let acc = F.fwd_join acc init MERGE in
      unroll acc post (nb+1)
            
    (* increasing iterations with join / widening *)
    and iter_up init inv nb =
      iter := Printf.sprintf "%s%02i_%s_" old_iter !n (if nb <= !nb_join then "join" else "widen"); incr n;
      let post = body ((Loop (lbl,After !nb_unroll))::ctx) inv in
      let inv' = F.fwd_join inv post MERGE in
      add_inv_fwd lbl ((Loop (lbl,After !nb_unroll))::ctx) inv';
      if F.subseteq inv' inv then iter_down init inv' 1 else
      let inv'' = if nb <= !nb_join then inv' else F.fwd_join inv inv' WIDEN in
      if !trace_lfp then (
        if nb <= !nb_join then Format.fprintf !log "lfp:join %i / %i@\n" nb !nb_join
        else Format.fprintf !log "lfp:widening %i@\n" (nb - !nb_join);
        Format.fprintf !log "%a@\n" F.print inv''
       );
      iter_up init inv'' (nb+1)

    (* decreasing iterations *)
    and iter_down init inv nb =
      iter := Printf.sprintf "%s%02i_down_" old_iter !n; incr n;
      if nb > !nb_down then inv else
      let post = body ((Loop (lbl,After !nb_unroll))::ctx) inv in
      let inv' = F.fwd_join init post MERGE in
      add_inv_fwd lbl ((Loop (lbl,After !nb_unroll))::ctx) inv';
      if !trace_lfp then
        Format.fprintf !log "lfp:decreasing %i / %i@\n:%a@\n" nb !nb_down F.print inv';
      if F.subseteq inv inv' then inv
      else iter_down init inv' (nb+1)

    in
    let r = unroll init init 1 in
    if !trace_lfp then Format.fprintf !log "lfp:loop invariant\n:%a@\n" F.print r;
    iter := old_iter;
    r
    

  (* under-approximation of body's greatest fixpoint
     1) perform some iterations with meet
     2) iterate with lower widening until stabilization
     3) unroll
     return 1 U 4

     TODO: 0) unroll last iterati
   *)
 let bwd_gfp ctx lbl body exit =
    let old_iter = !iter in
    let n = ref 1 in

    (* decreasing iterations with lower widening *)
    let rec iter_down post pre nb =
      iter := Printf.sprintf "%s%02i_%s_" old_iter !n (if nb <= !nb_meet then "meet" else "widen"); incr n;
      let b = body ((Loop (lbl,After !nb_unroll))::ctx) post exit pre in
      add_inv_bwd lbl ((Loop (lbl,After !nb_unroll))::ctx) b;
      if F.subseteq post b then (
        unroll b !nb_unroll 
      ) 
      else
      let c = F.bwd_meet post b (if nb <= !nb_meet then MERGE else WIDEN) in
      if !trace_gfp then (
        if nb <= !nb_meet then Format.fprintf !log "gfp:meet %i / %i@\n" nb !nb_meet
        else Format.fprintf !log "gfp:lower widening %i@\n" (nb - !nb_meet);
        Format.fprintf !log "%a@\n" F.print c;
        graph_debug (Printf.sprintf "%i_%s" nb (if nb <= !nb_meet then "meet" else "widen")) c;
       );
      iter_down c pre (nb+1)

    (* unrolling *)
    and unroll post nb =
      iter := Printf.sprintf "%s%02iunroll_" old_iter !n; incr n;
      if nb = 0 then post else
      let pre = get_inv_fwd lbl ((Loop (lbl,Unroll nb))::ctx) in
      let b = body ((Loop (lbl,Unroll nb))::ctx) post exit pre in
      add_inv_bwd lbl ((Loop (lbl,Unroll nb))::ctx) b;
      if !trace_gfp then
        Format.fprintf !log "gfp:unrolling %i / %i@\n:%a@\n" nb !nb_unroll F.print b;
      unroll b (nb-1)
    in

    let ctx = (Loop (lbl,After !nb_unroll))::ctx in
    let inv = get_inv_fwd lbl ctx in
    let r = iter_down inv inv 1 in
    if !trace_gfp then Format.fprintf !log "gfp:loop invariant@\n%a@\n" F.print r;
    iter := old_iter;
    r


  (* forward iterator *)
  (* **************** *)


  let rec fwd_stat (st:stack) (pre:F.t) (s:stat) (x:extent): F.t =
    match s with

    | T_expr e -> 
        if !trace_fwd then Format.fprintf !log "FWD %a@\n" print_expr e;
        let post,err = F.fwd_assign pre (WEAK []) e in
        post

    | T_assign ((v,vx),e) ->
        if !trace_fwd then Format.fprintf !log "FWD %a = %a@\n" print_var_name v print_expr e;
       let post,err = F.fwd_assign pre (STRONG v) e in
        post

    | T_call (f,fx) ->
        if !trace_fwd then Format.fprintf !log "FWD call %a@\n" print_func_name f;
        if List.mem (call_ctx f) st then failwith "recursive call";
        let pre, saved = F.get_conts pre in
        let post = fwd_block ((call_ctx f)::st) pre f.func_body in
        F.put_conts post saved

    | T_if (e,b1,b2) ->
        if !trace_fwd then Format.fprintf !log "FWD if (%a)@\n" print_expr e;
        let branch1,branch2,err = F.fwd_filter pre e in
        if !trace_fwd then Format.fprintf !log "FWD then (%a)@\n" print_expr e;
        let post1 = fwd_block st branch1 b1 in
        if !trace_fwd then Format.fprintf !log "FWD else (%a)@\n" print_expr e;
        let post2 = fwd_block st branch2 b2 in
        if !trace_fwd then Format.fprintf !log "FWD if merge@\n";
        F.fwd_join post1 post2 MERGE

    | T_while (lbl,e,b) ->
        let inv =
          fwd_lfp st lbl
            (fun st inv ->
              if !trace_fwd then Format.fprintf !log "FWD while (%a)@\n" print_expr e;
              let prebody,_,err = F.fwd_filter inv e in
              if !trace_fwd then Format.fprintf !log "FWD while body@\n";
              fwd_block st prebody b
            )
            pre
        in
        if !trace_fwd then Format.fprintf !log "FWD while (%a) exit@\n" print_expr e;
        let _,post,err = F.fwd_filter inv e in
        post
          
    | T_add_var (v, None) ->
        if !trace_fwd then Format.fprintf !log "FWD add_var %a@\n" print_var_name v;
        F.fwd_add_var pre v

    | T_add_var (v, Some e) ->
        if !trace_fwd then Format.fprintf !log "FWD add_var %a@\n" print_var_name v;
        let post = F.fwd_add_var pre v in
        if !trace_fwd then Format.fprintf !log "FWD add_var init %a@\n" print_expr e;
        let post,err = F.fwd_assign post (STRONG v) e in
        post

    | T_del_var v ->
        if !trace_fwd then Format.fprintf !log "FWD del_var %a@\n" print_var_name v;
        F.fwd_del_var pre v

    | T_RETURN ->
        if !trace_fwd then Format.fprintf !log "FWD return@\n";
        F.fwd_goto pre (goto_return st)

    | T_BREAK ->
        if !trace_fwd then Format.fprintf !log "FWD break@\n";
        F.fwd_goto pre (goto_break st)

    | T_label (s,_) ->
        if !trace_fwd then Format.fprintf !log "FWD label %s@\n" s;
        F.fwd_label pre s

    | T_assert e ->
        if !trace_fwd then Format.fprintf !log "FWD assert (%a) exit@\n" print_expr e;
        let branch1,branch2,err = F.fwd_filter pre e in
        if not (F.is_bot branch2) then
          Format.fprintf !log "WARNING %a: assert failure %a\n" 
            print_extent x print_expr e;
        branch1

    | T_assume e ->
        if !trace_fwd then Format.fprintf !log "FWD assume (%a) exit@\n" print_expr e;
        let branch1,branch2,err = F.fwd_filter pre e in
        branch1

    | T_print l ->
        if !enable_print then print_svg_file "output.svg" pre (List.map fst l);
        pre

  and fwd_block (st:stack) (pre:F.t) (b:block) : F.t =
    match b with
    | T_empty lbl -> 
        if !trace_fwd then 
          Format.fprintf !log "FWD %a:@\n%a@\n" print_point (lbl,st) F.print pre;
        add_inv_fwd lbl st pre; 
        pre
    | T_stat (lbl,(s,sx),b) ->
        if !trace_fwd then 
          Format.fprintf !log "FWD %a:@\n%a@\n" print_point (lbl,st) F.print pre;
        add_inv_fwd lbl st pre;
        fwd_block st (fwd_stat st pre s sx) b



  (* backward iterator *)
  (* ***************** *)


  (* returns a refined pre, given an original pre and refined post *)

  let rec bwd_stat (st:stack) (pre:F.t) (post:F.t) (s:stat) (x:extent): F.t =
    match s with

    | T_expr e ->
        if !trace_bwd then Format.fprintf !log "BWD %a@\n" print_expr e;
        F.bwd_assign post noerr (WEAK []) e pre

    | T_assign ((v,vx),e) ->
        if !trace_bwd then Format.fprintf !log "BWD %a = %a@\n" print_var_name v print_expr e;
       F.bwd_assign post noerr (STRONG v) e pre

    | T_call (f,fx) ->
        if !trace_bwd then Format.fprintf !log "BWD call %a@\n" print_func_name f;
        if List.mem (call_ctx f) st then failwith "recursive call";
        let post, saved = F.get_conts post in
        let mid = bwd_block ((call_ctx f)::st) post f.func_body in
        let mid = F.put_conts mid saved in
        F.bwd_meet mid pre MERGE

    | T_if (e,b1,b2) ->
        let branch1 = bwd_block st post b1 in
        let branch2 = bwd_block st post b2 in
        if !trace_bwd then Format.fprintf !log "BWD if (%a) merge\n" print_expr e;
        F.bwd_filter branch1 branch2 noerr e IFTHENELSE pre

    | T_while (lbl,e,b) ->
if !trace_bwd then Format.fprintf !log "start loop@\npost = %a@\npre = %a@\n" F.print post F.print pre;
let iter = ref 0 in
        let inv =
          bwd_gfp st lbl
            (fun st post exit pre ->
if !trace_bwd then Format.fprintf !log "start iter@\n  post = %a@\n  exit = %a@\n  pre = %a@\n" F.print post F.print exit F.print pre;

              if !trace_bwd then Format.fprintf !log "BWD while body@\n";
              let prebody = bwd_block st post b in

if !trace_bwd then Format.fprintf !log "  prebody = %a@\n" F.print prebody;
              
              if !trace_bwd then Format.fprintf !log "BWD while (%a)\n" print_expr e;
              let pre = F.bwd_filter prebody exit noerr e LOOP pre in

              if !trace_bwd then (
                incr iter;
                graph_debug (Printf.sprintf "%i_a_exit" !iter) exit;
                graph_debug (Printf.sprintf "%i_b_post" !iter) post;
                graph_debug (Printf.sprintf "%i_c_prebody" !iter) prebody;
                graph_debug (Printf.sprintf "%i_d_pre" !iter) pre;
               );
              
              if !trace_bwd then Format.fprintf !log "  pre = %a@\nend iter@\n" F.print pre;

              pre
            )
            post
        in
        F.bwd_meet pre inv MERGE
        
    | T_add_var (v, None) ->
        if !trace_bwd then Format.fprintf !log "BWD add_var %a@\n" print_var_name v;
        F.bwd_add_var post v pre

    | T_add_var (v, Some e) ->
        (* reconstruct the itermediate "variable created but not initialized"
           state
         *)
        let mid = F.fwd_add_var pre v in
        if !trace_bwd then Format.fprintf !log "BWD add_var init %a@\n" print_expr e;
        let mid = F.bwd_assign post noerr (STRONG v) e mid in
        if !trace_bwd then Format.fprintf !log "BWD add_var %a@\n" print_var_name v;
        F.bwd_add_var mid v pre

    | T_del_var v ->
        if !trace_bwd then Format.fprintf !log "BWD del_var %a@\n" print_var_name v;
        F.bwd_del_var post v pre

    | T_RETURN ->
        if !trace_bwd then Format.fprintf !log "BWD return@\n";
        F.bwd_goto post (goto_return st) pre

    | T_BREAK ->
        if !trace_bwd then Format.fprintf !log "BWD break@\n";
        F.bwd_goto post (goto_break st) pre

    | T_label (s,_) ->
        if !trace_bwd then Format.fprintf !log "BWD label %s@\n" s;
        F.bwd_label post s pre

    | T_assert e ->
        if !trace_bwd then Format.fprintf !log "BWD assert %a@\n" print_expr e;
        F.bwd_filter post (F.bot post) noerr e IFTHENELSE pre

    | T_assume e ->
        if !trace_bwd then Format.fprintf !log "BWD assume %a@\n" print_expr e;
        (*F.bwd_filter post (F.top post) noerr e IFTHENELSE pre*)
        post

    | T_print _ ->
        post


  and bwd_block (st:stack) (post:F.t) (b:block) : F.t =
    match b with
    | T_empty lbl -> 
        let pre = get_inv_fwd lbl st in
        let pre = F.bwd_meet pre post MERGE in
        if !trace_bwd then 
          Format.fprintf !log "BWD %a:@\n%a@\n" print_point (lbl,st) F.print pre;
        add_inv_bwd lbl st pre;
        pre

    | T_stat (lbl,(s,sx),b) ->
        let post = bwd_block st post b in
        let pre = get_inv_fwd lbl st in
        let pre = bwd_stat st pre post s sx in
        if !trace_bwd then 
          Format.fprintf !log "BWD %a:@\n%a@\n" print_point (lbl,st) F.print pre;
        add_inv_bwd lbl st pre;
        pre



  (* combined analyzer *)
  (* ***************** *)

  (* analyze the program, starting at the specified entry point *)
  let analyze ((init,funcs,vars):prog) (main:string) =
    latex_out := Some (open_out "out/output.tex");
    all_vars := vars;
    if not (StringMap.mem main funcs) then
      failwith ("cannot find entry-point: "^main);
    let f = StringMap.find main funcs in
    let vars = 
      (match f.func_return with Some v -> [v] | None -> [])@f.func_args 
    in
    Format.fprintf !log "@\nINITIALIZATION:@\n===============@\n";
    let a0 = fwd_block [] (F.empty ()) init in
    let a1 = List.fold_left (fun a v ->  F.fwd_add_var a v) a0 vars in
    Format.fprintf !log "@\n@\nFORWARD ANALYSIS:@\n=================@\n";
    let a2 = fwd_block [call_ctx f] a1 f.func_body in
    Format.fprintf !log "@\nforward invariant:@\n------------------@\n";
    if !dump_inv then print_inv_fwd !log;
    Format.fprintf !log "@\nforward result:@\n--------------@\n%a@\n" F.print a2;
    Format.fprintf !log "@\n@\nBACKWARD ANALYSIS:@\n==================@\n";
    let a3 = bwd_block [call_ctx f] a2 f.func_body in
    Format.fprintf !log "@\nbackward invariant:@\n-------------------@\n";
    if !dump_inv then print_inv_bwd !log;
    Format.fprintf !log "@\nbackward result:@\n----------------@\n%a@\n" F.print a3

end

