(*   
   Main entry point.

   Copyright (C) 2011 Antoine Miné
*)

let preprocess = ref ""
let filenames = ref []
let domain = ref "poly"
let main = ref "main"


(* parsing *)

(* filename: name of file to parse
   preprocess: if non-null, calls "preprocess filename" and parse the output
 *)
let parse_file preprocess filename =
  Lexer.use_line_directives := preprocess <> "";
  let f = 
    if preprocess = "" then open_in filename
    else Unix.open_process_in (preprocess^" "^(Filename.quote filename))
  in
  let lex = Lexing.from_channel f in
  try
    lex.Lexing.lex_curr_p <- 
      { lex.Lexing.lex_curr_p with Lexing.pos_fname = filename; };
    let r = Parser.file Lexer.start_token lex in
    if preprocess = "" then close_in f
    else ignore (Unix.close_process_in f);
    r
  with
  | Parser.Error ->
     Printf.eprintf "Parse error (invalid syntax) near %s\n" 
        (Abstract_syntax.string_of_position lex.Lexing.lex_start_p);
      failwith "Parse error"
  | Failure "lexing: empty token" ->
     Printf.eprintf "Parse error (invalid token) near %s\n" 
        (Abstract_syntax.string_of_position lex.Lexing.lex_start_p);
      failwith "Parse error"



(* command line *)

let parse_args () =
  let rec doit args = match args with
  | "-pp"::x::r -> preprocess := !preprocess^" "^x; doit r
  | "-main"::x::r -> main := x; doit r
  | "-domain"::x::r -> domain := x; doit r
  | "-unroll"::x::r -> Iterator.nb_unroll := int_of_string x; doit r
  | "-join"::x::r -> Iterator.nb_join := int_of_string x; doit r
  | "-down"::x::r -> Iterator.nb_down := int_of_string x; doit r
  | "-meet"::x::r -> Iterator.nb_meet := int_of_string x; doit r
  | x::r -> filenames := x::(!filenames); doit r
  | [] -> ()
  in
  doit (List.tl (Array.to_list Sys.argv));
  filenames := List.rev !filenames


(* create analyzer instance *)

module PolyAnalyzer = Iterator.Iterator(Apron_domain.PolkaDomain)
module OctAnalyzer = Iterator.Iterator(Apron_domain.OctagonDomain)


(* do all *)

let doit () = 
  parse_args ();
  if !filenames = [] then invalid_arg "no source file specified";
  let sources = List.map (parse_file !preprocess) !filenames in
  let program = Abstract_to_typed_syntax.translate_program sources in
  Typed_syntax.print_prog Format.std_formatter program;
  match !domain with
  | "oct" -> OctAnalyzer.analyze program !main
  | "poly" -> PolyAnalyzer.analyze program !main
  | _ -> invalid_arg "unknown domain"

let _ = doit ()
