(*   
   Menhir parser for banal's simple language.

   This is mostly a very simple subset of C.

   There should be one conflict, for else in nested if.

   Copyright (C) 2011 Antoine Miné
*)

%{
open Abstract_syntax
%}

/* tokens */
/**********/

%token TOK_BOOL
%token TOK_CHAR
%token TOK_SHORT
%token TOK_INT
%token TOK_LONG
%token TOK_INTEGER
%token TOK_UNSIGNED
%token TOK_FLOAT
%token TOK_DOUBLE
%token TOK_REAL
%token TOK_VOID
%token TOK_WHILE
%token TOK_IF
%token TOK_ELSE
%token TOK_RETURN
%token TOK_BREAK
%token TOK_TRUE
%token TOK_FALSE
%token TOK_INPUT
%token TOK_VOLATILE
%token TOK_ASSERT
%token TOK_ASSUME
%token TOK_PRINT

%token TOK_LPAREN
%token TOK_RPAREN
%token TOK_LBRACKET
%token TOK_RBRACKET
%token TOK_LCURLY
%token TOK_RCURLY
%token TOK_PLUS_PLUS
%token TOK_MINUS_MINUS
%token TOK_STAR
%token TOK_PLUS
%token TOK_MINUS
%token TOK_EXCLAIM
%token TOK_DIVIDE
%token TOK_PERCENT
%token TOK_LESS
%token TOK_GREATER
%token TOK_LESS_EQUAL
%token TOK_GREATER_EQUAL
%token TOK_EQUAL_EQUAL
%token TOK_NOT_EQUAL
%token TOK_AND_AND
%token TOK_BAR_BAR
%token TOK_SEMICOLON
%token TOK_COLON
%token TOK_EQUAL
%token TOK_STAR_EQUAL
%token TOK_DIVIDE_EQUAL
%token TOK_PERCENT_EQUAL
%token TOK_PLUS_EQUAL
%token TOK_MINUS_EQUAL
%token TOK_COMMA

%token <string> TOK_id
%token <string> TOK_int_literal
%token <string> TOK_float_literal

%token TOK_EOF

/* priorities of binary operators (lowest to highest) */
%left TOK_BAR_BAR
%left TOK_AND_AND
%left TOK_EQUAL_EQUAL TOK_NOT_EQUAL
%left TOK_LESS TOK_GREATER TOK_LESS_EQUAL TOK_GREATER_EQUAL
%left TOK_PLUS TOK_MINUS
%left TOK_STAR TOK_DIVIDE TOK_PERCENT


/* entry-points */
/****************/

%start<Abstract_syntax.decl list Abstract_syntax.ext> file

%%


/* toplevel */
/************/

file: t=ext(list(decl)) TOK_EOF { t }


/* expressions */
/***************/

primary_expr:
| TOK_LPAREN e=expr TOK_RPAREN     { e }
| e=TOK_id                         { A_identifier e }
| e=TOK_int_literal                { A_int_const e }
| e=TOK_float_literal              { A_float_const e }
| TOK_TRUE                         { A_bool_const true }
| TOK_FALSE                        { A_bool_const false }
| TOK_LBRACKET  e1=ext(sign_int_literal)  
  TOK_SEMICOLON e2=ext(sign_int_literal) TOK_RBRACKET
  { A_int_itv (e1, e2) }
| TOK_LBRACKET  e1=ext(sign_float_literal) 
  TOK_SEMICOLON e2=ext(sign_float_literal) TOK_RBRACKET
| TOK_LBRACKET  e1=ext(sign_float_literal) 
  TOK_SEMICOLON e2=ext(sign_int_literal) TOK_RBRACKET
| TOK_LBRACKET  e1=ext(sign_int_literal) 
  TOK_SEMICOLON e2=ext(sign_float_literal) TOK_RBRACKET
  { A_float_itv (e1, e2) }

sign_int_literal:
| i=TOK_int_literal            { i }
| TOK_PLUS i=TOK_int_literal   { i }
| TOK_MINUS i=TOK_int_literal  { "-"^i }

sign_float_literal:
| f=TOK_float_literal            { f }
| TOK_PLUS f=TOK_float_literal   { f }
| TOK_MINUS f=TOK_float_literal  { "-"^f }


postfix_expr:
| e=primary_expr         { e }
| l=ext(lvalue) o=incr   { A_increment (l, o, A_POST) }
| e=ext(TOK_id) TOK_LPAREN l=separated_list(TOK_COMMA,ext(expr)) 
  TOK_RPAREN 
  { A_call (e, l) }

%inline incr:
| TOK_PLUS_PLUS    { A_INCR }
| TOK_MINUS_MINUS  { A_DECR }

unary_expr:
| e=postfix_expr                   { e }
| o=incr l=ext(lvalue)             { A_increment(l, o, A_PRE) }
| o=unary_op e=ext(cast_expr)      { A_unary (o, e) }

%inline unary_op:
| TOK_PLUS           { A_UNARY_PLUS }
| TOK_MINUS          { A_UNARY_MINUS }
| TOK_EXCLAIM        { A_NOT }

cast_expr:
| e=unary_expr                                        { e }
| TOK_LPAREN t=ext(typ) TOK_RPAREN e=ext(cast_expr)   { A_unary (A_cast t, e) }

binary_expr:
| e=cast_expr                                         { e }
| e=ext(binary_expr) o=binary_op f=ext(binary_expr)   { A_binary (o, e, f) }

%inline binary_op:
| TOK_STAR           { A_MULTIPLY }
| TOK_DIVIDE         { A_DIVIDE }
| TOK_PERCENT        { A_MODULO }
| TOK_PLUS           { A_PLUS }
| TOK_MINUS          { A_MINUS }
| TOK_LESS           { A_LESS }
| TOK_GREATER        { A_GREATER }
| TOK_LESS_EQUAL     { A_LESS_EQUAL }
| TOK_GREATER_EQUAL  { A_GREATER_EQUAL }
| TOK_EQUAL_EQUAL    { A_EQUAL }
| TOK_NOT_EQUAL      { A_NOT_EQUAL }
| TOK_AND_AND        { A_AND }
| TOK_BAR_BAR        { A_OR }

expr:
| e=binary_expr                           { e }
| e=ext(lvalue) o=assign_op f=ext(expr)   { A_assign (e, o, f) }

%inline assign_op:
| TOK_EQUAL          { None }
| TOK_STAR_EQUAL     { Some A_MULTIPLY_ASSIGN }
| TOK_DIVIDE_EQUAL   { Some A_DIVIDE_ASSIGN }
| TOK_PERCENT_EQUAL  { Some A_MODULO_ASSIGN }
| TOK_PLUS_EQUAL     { Some A_PLUS_ASSIGN }
| TOK_MINUS_EQUAL    { Some A_MINUS_ASSIGN }

lvalue:
| i=TOK_id  { i }


/* declarations */
/****************/

decl:
| d=ext(var_decl)       { A_global (d, A_VARIABLE) }
| d=ext(input_decl)     { A_global (d, A_INPUT) }
| d=ext(volatile_decl)  { A_global (d, A_VOLATILE) }
| d=ext(fun_decl)       { A_function d }

input_decl:
| TOK_INPUT v=var_decl { v }

volatile_decl:
| TOK_VOLATILE v=var_decl { v }

var_decl:
| s=ext(typ) i=separated_list(TOK_COMMA,init_declarator) TOK_SEMICOLON
  { s, i }

init_declarator:
| v=ext(TOK_id)                         { v, None }
| v=ext(TOK_id) TOK_EQUAL i=ext(expr)   { v, Some i }

fun_decl:
| t=typ_void i=ext(TOK_id) TOK_LPAREN p=separated_list(TOK_COMMA,param_decl) 
  TOK_RPAREN b=block
  { t, i, p, b }

param_decl:
| s=ext(typ) v=ext(TOK_id) { v, s }

%inline typ_void:
| t=ext(typ)   { Some t }
| TOK_VOID     { None }

typ:
| s=typ_sign t=typ_int   { A_int (t,s) }
| t=typ_float            { A_float t }
| TOK_BOOL               { A_BOOL }

typ_float:
| TOK_FLOAT    { A_FLOAT }
| TOK_DOUBLE   { A_DOUBLE }
| TOK_REAL     { A_REAL }

typ_sign:
|              { A_SIGNED }
| TOK_UNSIGNED { A_UNSIGNED }

typ_int:
| TOK_CHAR     { A_CHAR }
| TOK_SHORT    { A_SHORT }
| TOK_INT      { A_INT }
| TOK_LONG     { A_LONG }
| TOK_INTEGER  { A_INTEGER }


/* statements */
/**************/

block:
| TOK_LCURLY l=list(ext(stat)) TOK_RCURLY  { l }

stat:
| l=block                     { A_block l }
| TOK_SEMICOLON               { A_SKIP }
| s=ext(expr) TOK_SEMICOLON   { A_expr s }
| TOK_IF TOK_LPAREN e=ext(expr) TOK_RPAREN s=ext(stat)
  { A_if (e, s, None) }
| TOK_IF TOK_LPAREN e=ext(expr) TOK_RPAREN s=ext(stat) TOK_ELSE t=ext(stat) 
  { A_if (e, s, Some t) }
| TOK_WHILE TOK_LPAREN e=ext(expr) TOK_RPAREN s=ext(stat)
  { A_while (e, s) }
| TOK_RETURN e=option(ext(expr)) TOK_SEMICOLON
  { A_return e }
| TOK_BREAK
  { A_BREAK }
| v=var_decl
  { A_local v }
| l=ext(TOK_id) TOK_COLON
  { A_label l }
| TOK_ASSERT TOK_LPAREN e=ext(expr) TOK_RPAREN TOK_SEMICOLON
  { A_assert e }
| TOK_ASSUME TOK_LPAREN e=ext(expr) TOK_RPAREN TOK_SEMICOLON
  { A_assume e }
| TOK_PRINT TOK_LPAREN l=separated_list(TOK_COMMA,ext(lvalue)) TOK_RPAREN
  { A_print l }


/* utilities */
/*************/

/* adds extent information to rule */
%inline ext(X): 
| x=X { x, ($startpos, $endpos) }


%%

