(*   
     Date management.
     Based on simplified RFC 1123 & 1036.
     
     Internally, all dates are in GMT.
     
     Copyright (C) 2012 Antoine Miné
 *)


open Unix


let days = 
  [|"Sun";"Mon";"Tue";"Wed";"Thu";"Fri";"Sat"|]

let months = 
  [|"Jan";"Feb";"Mar";"Apr";"May";"Jun";"Jul";"Aug";"Sep";"Oct";"Nov";"Dec"|]


(* month name -> month number (0..11) *)
let get_month (s:string) = 
  try match String.lowercase (String.sub s 0 3) with
    "jan" -> 0 | "feb" -> 1 | "mar" -> 2 |"apr" -> 3 | "may" -> 4 | "jun" -> 5
  | "jul" -> 6 | "aug" -> 7 | "sep" -> 8 |"oct" -> 9 | "nov" -> 10 | "dec" -> 11
  | _ -> raise Not_found
  with _ -> raise(Invalid_argument("get_month: "^s))


(* delta between GMT and local time *)
let local_minus_gmt = fst (mktime (gmtime    1000000.)) -.
                      fst (mktime (localtime 1000000.))


(* date -> string representation (following RFC 1123) *)
let to_string_tm (t:tm) : string =
  Printf.sprintf "%s, %02d %s %04d %02d:%02d:%02d GMT"
    days.(t.tm_wday) t.tm_mday months.(t.tm_mon) 
    (t.tm_year+1900) t.tm_hour t.tm_min t.tm_sec

let to_string d = to_string_tm (gmtime d)
let to_string_local d = to_string_tm (localtime d)


(* compact string representation (for log) *)
let to_string_compact_tm (t:tm) : string =
  Printf.sprintf "%02d-%02d-%04d %02d:%02d:%02d"
    t.tm_mday (t.tm_mon+1) (t.tm_year+1900)
    t.tm_hour t.tm_min t.tm_sec

let to_string_compact d = to_string_compact_tm (gmtime d)
let to_string_compact_local d = to_string_compact_tm (localtime d)


let date_sep = Str.regexp "[ :,-]+"

(* parse RFC 1123 & 1036 dates *)
let parse (s:string) : tm =
  match Str.split date_sep s with
  | _::mday::month::year::hour::min::sec::_ ->
      let y = int_of_string year in
      let y = if y>100 then y-1900 else y in
      { tm_mday = int_of_string mday;
        tm_mon  = get_month month;
        tm_year = y;
        tm_hour = int_of_string hour;
        tm_min  = int_of_string min;
        tm_sec  = int_of_string sec;
        tm_wday = -1;     (* ignored my mktime *)
        tm_yday = -1;     (* '' *)
        tm_isdst = false; (* '' *)
      }
  | _ -> raise(Invalid_argument("string_of_date: "^s))


let of_string (s:string) : float =
  fst (mktime (parse s)) -. local_minus_gmt

let of_local_string (s:string) : float =
  fst (mktime (parse s))


(* current date and time *)
let current () =
  time ()



(* to use whith %a in printf *)
let pp f date =
  output_string f (to_string_compact date)

(* to use whith %t in printf *)
let pp_cur f =
  output_string f (to_string_compact (current ()))


(* to use whith %a in printf *)
let pp_local f date =
  output_string f (to_string_compact_local date)

(* to use whith %t in printf *)
let pp_cur_local f =
  output_string f (to_string_compact_local (current ()))


