(*   
     Web-based analyzer.
     
     Copyright (C) 2012 Antoine Miné
*)

open Http
open Html

open Datatypes
open Abstract_syntax
open Typed_syntax


let static_root = ref "/~mine/banal"

(* limits *)
let max_prog_length = 4096
let max_vars = 12



(* printers *)
(* -------- *)


let pp_char (h:http) = function
  | ' ' -> hstring h "&nbsp;"
  | '\n' -> hstring h "<br>\r\n"
  | '\r' -> ()
  | '<' -> hstring h "&lt;"
  | '>' -> hstring h "&gt;"
  | '&' -> hstring h "&amp;"
  | '\"' -> hstring h "&quot;"
  | x -> hchar h x


(* prints a progrm source, with possible highlight (e.g., to show an error) *)
let print_source (h:http) (s:string) (pos1:int option) (pos2:int option) =
  for i=0 to (String.length s)-1 do
    if pos1 = Some i then hstring h "<span class='error'>";
    if pos2 = Some i then hstring h "</span>";
    pp_char h s.[i]
  done
  

(* prints a program from its syntax tree, in HTML with coloring *)
let print_prog h (init,funcs,vars) =
  let o = h.http_out in
  let span style o s =
    Printf.fprintf o "<span class='%s'>%a</span>" style html_escape s
  in
  let rec var_name o v = span "var" o v.var_name
  and func_name o f = span "func" o f.func_name
  and list f sep o = function
  | [] -> ()
  | [a] -> f o a
  | a::b -> f o a; output_string o sep; list f sep o b
  and label o (l,p) = 
    let id = (string_of_id "" l) in
    Printf.fprintf o 
      "{<a class='pt' name='pt-%s'>%s</a>|<a href='#fwd-%s'>f</a>|<a href='#bwd-%s'>b</a>} " 
      id id id id
  and expr o ((ee,_,_) as e) = match ee with
  | T_unary (op,e1) ->
      span "op" o (string_of_unary_op op);
      if expr_precedence e1 <= expr_precedence e
      then Printf.fprintf o " (%a)" expr e1
      else Printf.fprintf o " %a" expr e1
  | T_binary (op,e1,e2) ->
      if expr_precedence e1 < expr_precedence e
      then Printf.fprintf o "(%a) " expr e1
      else Printf.fprintf o "%a " expr e1;
      span "op" o  (string_of_binary_op op);
      if expr_precedence e2 <= expr_precedence e
      then Printf.fprintf o " (%a)" expr e2
      else Printf.fprintf o " %a" expr e2          
  | T_float_const f -> span "num" o (string_of_float_set f)
  | T_int_const i -> span "num" o (string_of_int_set i)
  | T_bool_const b -> span "bool" o (string_of_tbool b)
  | T_var v -> var_name o v
  and stat lbl ind o s = match s with
  | T_expr e -> Printf.fprintf o "%a" expr e
  | T_assign ((v,_),e) -> Printf.fprintf o "%a = %a" var_name v expr e
  | T_call (f,fx) ->
      (match f.func_return with
      | Some v -> Printf.fprintf o "%a = " var_name v
      | None -> ());
      Printf.fprintf o "%a(" func_name f;
      list var_name "," o f.func_args;
      output_string o ")"
  | T_if (e, b1, b2) ->
      Printf.fprintf o "%a (%a)<br>\r\n" (span "key") "if" expr e;
      block lbl ind o b1;
      Printf.fprintf o "%s%a<br>\r\n" ind (span "key") "else";
      block lbl ind o b2
  | T_while (l,e,b) ->
      Printf.fprintf o "%a %a (%a)<br>\r\n" (span "key") "while" label l expr e;
      block lbl ind o b
  | T_add_var (v, eo) ->
      Printf.fprintf o "%a%s %a" 
        (span "key")
        (match v.var_scope with
        | T_INPUT -> "input "
        | T_VOLATILE -> "volatile "
        | _ -> "")
        (string_of_typ v.var_typ) 
        var_name v;
      (match eo with
      | Some e -> Printf.fprintf o " = %a" expr e
      | None -> ())        
  | T_del_var v ->
      Printf.fprintf o "%a %a" (span "key") "delete" var_name v
  | T_RETURN -> span "key" o "return"
  | T_BREAK -> span "key" o "break"
  | T_label (s,_) -> span "lbl" o (s^":")
  | T_assert e ->
      Printf.fprintf o "%a (%a)" (span "key") "assert" expr e
  | T_assume e ->
      Printf.fprintf o "%a (%a)" (span "key") "assume" expr e
  | T_print l ->
      Printf.fprintf o "%a (%a)" (span "key") "print" (list var_name ",") (List.map fst l)
  and stat_end lbl ind o s =
    stat lbl ind o s;
    (match s with T_if _ | T_while _ -> () | _ -> Printf.fprintf o ";<br>\r\n")
  and block lbl ind o s =
    let ind2 = ind ^ "&nbsp;&nbsp;" in
    let rec aux = function
      | T_stat (l, (s,_), r) ->
          output_string o ind2;
          if lbl then label o l;
          stat_end lbl ind2 o s;
          aux r
      | T_empty l -> 
          output_string o ind2;
          if lbl then label o l;
          output_string o "<br>\r\n"
    in
    Printf.fprintf o "%s{<br>\r\n" ind;
    aux s;
    Printf.fprintf o "%s}<br>\r\n" ind
  and func o f =
    (match f.func_return with
    | None -> output_string o "void"
    | Some v -> html_escape o (string_of_typ v.var_typ));
    Printf.fprintf o " %a(" func_name f;
    list 
      (fun o v -> 
        Printf.fprintf o "%s %a" (string_of_typ v.var_typ) var_name v
      ) 
      "," o f.func_args;
    output_string o ")<br>\r\n";
    block true "" o f.func_body 
  in
  output_string o "<div class='oklog'>";
  block false "" o init;
  StringMap.iter (fun _ -> func o) funcs;
  output_string o "</div>\r\n";
  flush o
    


(* analyzer *)
(* -------- *)


module A = Iterator.Iterator(Apron_domain.PolkaDomain)


let get_val (h:http) (key:string) : string =
  try List.assoc key h.http_post with Not_found -> ""

let get_int (h:http) (key:string) (def:int) : int =
  try 
    let i = int_of_string (List.assoc key h.http_post) in
    if i >= 0 then i else def
  with _ -> def
  



let print_inv (h:http) (pref:string) (xy:(var*var) option) inv =
  let idfound = Hashtbl.create 10 in
  let rec print_stack = function
    | [] -> ()
    | [a] -> hprintf h "%a" html_escape (A.string_of_ctx a)
    | a::b -> hprintf h "%a:" html_escape (A.string_of_ctx a); print_stack b
  in
  let print_point ((id,pos),stack) =
    let id = string_of_id "" id in
    if not (Hashtbl.mem idfound id) then (
      Hashtbl.add idfound id ();
      hprintf h "<div class='ptinv'><a name='%s-%s'>Point <a href='#pt-%s'>{%s}</a></a></div>\r\n" pref id id id
     );
    hstring h "<div class='ptinv2'>stack: ["; print_stack stack; hstring h "]</div>\r\n"
  in
  let nb = ref 0 in
  List.iter 
    (fun (p,a) -> 
      (* do not print global initialization *)
      if snd p <> [] then (
        print_point p;
        hprintf h "<div class='inv'>%a</div>\r\n" A.F.print_html a;
        (match xy with
        | Some xy -> 
            hprintf h "<div id='graph-%s-%i' class='graph'>\r\n" pref !nb;
            A.F.print_svg h.http_out ~color:true (240,240) a xy;
            hprintf h "</div>\r\n";
            incr nb
        | None -> ());
       );
      hflush h
    ) 
    inv


let analyze (h:http) ((init,funcs,vars):prog) (main:string) (session:string) =
  let fmt = html_formatter h in
  Iterator.log := fmt;
  Iterator.trace_lfp := false; Iterator.trace_gfp := false; 
  Iterator.trace_fwd := false; Iterator.trace_bwd := false;
  Iterator.trace_iter := false;
  Iterator.enable_print := false;
  Iterator.nb_unroll := get_int h "unroll" 2;
  Iterator.nb_join := get_int h "join" 2;
  Iterator.nb_down := get_int h "down" 2;
  Iterator.nb_meet := get_int h "meet" 4;
  let f = try StringMap.find main funcs with Not_found -> failwith ("cannot find entry-point: "^main) in
  let vars = 
    (match f.func_return with Some v -> [v] | None -> [])@f.func_args 
  in
  hstring h "<p>Forward analysis...\r\n<div class='log'>"; hflush h;
  let a0 = A.fwd_block [] (A.F.empty ()) init in
  let a1 = List.fold_left (fun a v ->  A.F.fwd_add_var a v) a0 vars in
  let a2 = A.fwd_block [A.call_ctx f] a1 f.func_body in
  Format.pp_print_flush fmt ();
  hprintf h "<b>Invariant at program exit:</b> %a\r\n" A.F.print_html a2;
  hstring h "</div>\r\n";

  hstring h "<p>Backward analysis...\r\n<div class='log'>"; hflush h;
  let a3 = A.bwd_block [A.call_ctx f] a2 f.func_body in
  Format.pp_print_flush fmt ();
  hprintf h "<b>Sufficient condition at program input:</b> %a\r\n" A.F.print_html a3;
  hstring h "</div>\r\n";
  List.rev (A.InvMap.fold (fun a b l -> (a,b)::l) !A.fwd_inv_map []),
  A.InvMap.fold (fun a b l -> (a,b)::l) !A.bwd_inv_map []
  



(* driver *)
(* ------ *)


let finish h =
  hprintf h "<br><p><a href='%s/index.html#main'>Run another analysis</a>\r\n" !static_root;
  hprintf h "<hr>&copy; Antoine Miné 2012.<br>contact: <tt>mine</tt> at <tt>di.ens.fr</tt>.";
  hstring h "</body></html>";
  hflush h;
  raise Exit


let parse_prog h prog =
  let lex = Lexing.from_string prog in
  lex.Lexing.lex_curr_p <- 
    { lex.Lexing.lex_curr_p with Lexing.pos_fname = "input"; };
  try 
    let p = Parser.file Lexer.start_token lex in
    Abstract_to_typed_syntax.translate_program  [p]
  with
  | Parser.Error 
  | Failure "lexing: empty token" ->
      hprintf h "<p><div class='error'>Syntax error near <tt>%s</tt>.</div><p>The approximate position of the error is highlighted below:<br>" 
        (Abstract_syntax.string_of_position lex.Lexing.lex_start_p);
      hstring h "<div class='errlog'>";
      print_source h prog 
        (Some lex.Lexing.lex_start_pos) 
        (Some (lex.Lexing.lex_start_pos+1));
      hstring h "</div>\n";
      finish h 
  | Abstract_to_typed_syntax.Translate_error (x,s) ->
      hprintf h "<p><div class='error'>Syntax error near <tt>%s</tt>: %s.</div><p>The approximate position of the error is highlighted below:<br>"
        (Abstract_syntax.string_of_extent x) s;
      hstring h "<div class='errlog'>";
      print_source h prog 
        (Some (fst x).Lexing.pos_cnum) (Some (snd x).Lexing.pos_cnum);
      hstring h "</div>\n";
      finish h



let handle (h:http) =
  http_header h;
  try
    hprintf h "%s\r\n" doctype;
    hprintf h "<html>\r\n<header><title>Analysis results</title><link rel='stylesheet' type='text/css' href='%s/style.css'><script type='text/javascript' src='%s/analyzer.js'></script></header>\r\n<body>\r\n" !static_root !static_root;
    let prog = get_val h "program" in
    let main = get_val h "main" in
    if prog = "" then (hstring h "<p>No program!"; finish h);
    if main = "" then (hstring h "<p>No entry point!"; finish h);
    hstring h "<h2><a name='log'>Analysis log</a></h2>\r\n<p>Parsing...<br>\r\n"; hflush h;
    if String.length prog > max_prog_length then (hstring h "<p>Program too large."; finish h);
    let (init,funcs,vars) as program = parse_prog h prog in
    if IdMap.cardinal vars > max_vars then (hstring h "<p>Too many variables."; finish h);
    hstring h "<div class='log'>";
    print_source h prog None None;
    hstring h "</div>\r\n";
    let find_var name =
      IdMap.fold 
        (fun _ v acc -> if name=v.var_name then Some v else acc) vars None 
    in
    let xy = match find_var (get_val h "x"), find_var (get_val h "y") with
    | Some x, Some y -> Some (x,y)
    | _ -> None
    in
    let fwd, bwd = analyze h program main "session" in
    hstring h "<p>Analysis complete.";
    hstring h "<br><h2><a name='annotated'>Annotated program</a></h2>\r\n<p>Your program (after simplification) is shown below. Click on a program point to get the inferred forward invariant (f) or backward sufficient condition (b).<br>";
    print_prog h program;
    hstring h "<br><h2><a name='fwd'>Forward invariants</a></h2>\r\n";
    print_inv h "fwd" xy fwd;
    hstring h "<br><h2><a name='bwd'>Backward sufficient conditions</a></h2>\r\n";
    print_inv h "bwd" xy bwd;
    hprintf h "<div class='menu'><a href='#log'>Analysis log</a><br><a href='#annotated'>Annotated program</a><br><a href='#fwd'>Forward invariants</a><br><a href='#bwd'>Backward sufficient conditions</a><br><br><a href='%s/index.html#main'>Run another analysis</a></div>\r\n" !static_root;
    finish h
  with 
  | Exit -> ()
  | Timeout ->
      Printf.fprintf log "%t analysis timed out\n" log_banner;
      hprintf h "<p><div class='error'>The analysis timed out</div>\r\n";
      finish h
  | x ->
      Printf.fprintf log "%t exception %s\n" log_banner (Printexc.to_string x); 
      hprintf h "<p><div class='error'>The analyzer encountered an internal error: %a.</div>\r\n" html_escape (Printexc.to_string x);
      finish h
        


(* entry point *)
(* ----------- *)

let main () =
  let args = Array.to_list Sys.argv in
  if List.mem "--server" args then (
    static_root := "/static";
    http_server 8080 handle
   )
  else cgi_answerer handle

let _ = main ()

